<?php defined('BX_DOL') or die('hack attempt');

/**
 * Photos Auto Album alert handler.
 */
class BxPhotosAutoAlbumAlerts extends BxDolAlertsResponse
{
    /**
     * Collected uploads per profile for this request.
     * Structure: [profile_id => [ [name, size, ts], ... ]]
     * We intentionally collect metadata (not IDs) because system:store_file
     * alert fires before DB row is created, so file IDs are not available yet.
     * We'll resolve IDs at the end of request.
     * @var array<int, array<int, array{name:string,size:int,ts:int}>>
     */
    protected static $aUploads = [];

    /**
     * Ensure we register one shutdown handler per request.
     * @var bool
     */
    protected static $bShutdownRegistered = false;

    public function response($oAlert)
    {
        if ($oAlert->sUnit === 'system') {
            switch ($oAlert->sAction) {
                case 'store_file':
                    $this->handleStoreFile($oAlert);
                    break;
            }
        }

        if ($oAlert->sUnit === 'bx_photos' && $oAlert->sAction === 'added') {
            $this->handlePhotoAdded($oAlert);
        }
    }

    /**
     * Capture uploaded photos metadata (no IDs yet at this point).
     */
    protected function handleStoreFile(BxDolAlerts $oAlert)
    {
        $sStorage = $oAlert->aExtras['storage_object'] ?? '';
        // Only track Photos uploads (bx_photos_photos storage)
        if ($sStorage !== 'bx_photos_photos') {
            return;
        }

        $iProfileId = (int)$oAlert->iSender; // profile id from alert sender
        $sName = (string)($oAlert->aExtras['file_name'] ?? '');
        $iSize = (int)($oAlert->aExtras['size'] ?? 0);

        bx_log('bx_photos_autoalbum', "store_file: storage={$sStorage}, profile={$iProfileId}, name={$sName}, size={$iSize}");

        if (!$iProfileId || $sName === '' || $iSize <= 0) {
            return;
        }

        if (!isset(self::$aUploads[$iProfileId])) {
            self::$aUploads[$iProfileId] = [];
        }

        $iNow = time();
        self::$aUploads[$iProfileId][] = [
            'name' => $sName,
            'size' => $iSize,
            'ts'   => $iNow,
        ];

        // Persist to queue table to coordinate across parallel requests.
        $this->ensureQueueTable();
        $this->enqueueUpload($iProfileId, $sName, $iSize, $iNow);

        // No shutdown function needed anymore, album creation is triggered by 'bx_photos:added'
        // alert which fires after the main content form submission.
    }

    /**
     * Create album after page output if needed.
     */
    /**
     * Trigger album creation when a new photo entry is added.
     */
    protected function handlePhotoAdded(BxDolAlerts $oAlert): void
    {
        $iProfileId = (int)$oAlert->iSender;
        $iContentId = (int)$oAlert->iObject;
        $aExtras = $oAlert->aExtras;

        // The 'added' alert fires after the photo entry is created in the database.
        // We will now use the content ID to find the file ID and then use the queue
        // to find other concurrently uploaded files.
        bx_log('bx_photos_autoalbum', "photo_added: profile={$iProfileId}, content_id={$iContentId}, extras=" . var_export($aExtras, true));

        // Get the file ID of the photo that triggered this alert
        $oDb = BxDolDb::getInstance();
        $aPhoto = $oDb->getRow("SELECT * FROM `bx_photos_photos` WHERE `id` = ?", [$iContentId]);
        
        if (empty($aPhoto)) {
            bx_log('bx_photos_autoalbum', "handlePhotoAdded: Could not find photo entry with ID {$iContentId}");
            return;
        }

        // Use the file metadata of the newly added photo to search the queue for all related uploads
        $aUploads = [
            [
                'name' => $aPhoto['file_name'],
                'size' => $aPhoto['size'],
                'ts'   => $aPhoto['date'], // Use the photo's creation time as a reference
            ]
        ];

        // We rely on the queue processing logic in createAlbumFromRecentUploads to find the rest.
        $this->createAlbumFromRecentUploads($iProfileId, $aUploads);
    }

    /**
     * Resolve recent Photos uploads to file IDs, create an Album, copy files into Albums storage, and associate them.
     */
    protected function createAlbumFromRecentUploads(int $iProfileId, array $aUploads): void
    {
        // 1) Acquire lock to avoid duplicate album creation across parallel requests
        $this->ensureLockTable();
        if (!$this->acquireLock($iProfileId)) {
            bx_log('bx_photos_autoalbum', 'Another worker holds lock; skipping for profile=' . $iProfileId);
            return;
        }

        try {
            // 2) Resolve recent queue burst for this profile (cross-request)
            //    Include the freshest rows as well to catch parallel uploads; no grace cutoff.
            $aQueue = $this->fetchRecentQueue($iProfileId, 180); // Revert to 3 minutes, as the photo's date is now the anchor
            if (count($aQueue) <= 1) {
                bx_log('bx_photos_autoalbum', 'No burst in queue (<=1). queue_count=' . count($aQueue));
                return;
            }

            // 3) Translate queued items to actual file IDs in Photos storage
            $aPhotoMap = $this->resolveQueuedPhotoFileIds($iProfileId, $aQueue);
            $aValidIds = array_filter($aPhotoMap);
            
            // The photo that triggered the alert is guaranteed to be in the database,
            // but may not be in the queue if the alert fired before the queue entry was written.
            // We must ensure the photo that triggered the alert is included in the map.
            $aTriggerPhoto = $oDb->getRow("SELECT * FROM `bx_photos_photos` WHERE `id` = ?", [(int)$oAlert->iObject]);
            if (!empty($aTriggerPhoto)) {
                // Find the queue entry that corresponds to the trigger photo
                $iTriggerQueueId = 0;
                foreach ($aQueue as $q) {
                    if ($q['name'] === $aTriggerPhoto['file_name'] && (int)$q['size'] === (int)$aTriggerPhoto['size']) {
                        $iTriggerQueueId = (int)$q['id'];
                        break;
                    }
                }

                if ($iTriggerQueueId > 0) {
                    $aPhotoMap[$iTriggerQueueId] = (int)$oAlert->iObject;
                } else {
                    // This is a fallback in case the queue entry for the trigger photo is missing.
                    // We manually add a dummy entry to the map to ensure the trigger photo is processed.
                    // This is a last resort and indicates a deeper issue with the queue.
                    $aPhotoMap[0] = (int)$oAlert->iObject;
                }
            }

            $aValidIds = array_filter($aPhotoMap);
            
            if (count($aValidIds) <= 1) {
                bx_log('bx_photos_autoalbum', 'Burst not detected (<=1 recent photos). IDs=' . json_encode(array_values($aValidIds)));
                return;
            }

            $aSuccessQueueIds = [];
            $aFailedQueueIds = [];

            // 4) Skip if an auto-album was just created
            if ($this->recentAutoAlbumExists($iProfileId, 60)) {
                bx_log('bx_photos_autoalbum', 'Recent auto-album exists; skipping creation.');
                $this->deleteQueueEntries(array_column($aQueue, 'id'));
                $this->releaseLock($iProfileId); // Release lock early if skipping
                return;
            }

            // 5) Create Album entry via service (no redirect)
        $sDateDisplay = gmdate('M j, Y @ H:i:s');
        $sTitle = 'Upload from ' . $sDateDisplay;
        $sText = 'Photos uploaded on ' . $sDateDisplay . ' UTC.';

            $aVals = [
                'title' => $sTitle,
                'text' => $sText,
                'allow_view_to' => BX_DOL_PG_ALL,
                'album_category' => $this->determineAlbumCategory($iProfileId),
            ];

            $aRes = (array)BxDolService::call('bx_albums', 'entity_add', [$iProfileId, $aVals]);
            if (empty($aRes) || (int)$aRes['code'] !== 0 || empty($aRes['content']) || !is_array($aRes['content'])) {
                bx_log('bx_photos_autoalbum', 'Failed to create album via service: ' . var_export($aRes, true));
                return;
            }

            $iAlbumId = (int)$aRes['content']['id'];
            if (!$iAlbumId) {
                return;
            }

            // 6) Copy files into Albums storage and associate with album
            $oStorageAlbums = BxDolStorage::getObjectInstance('bx_albums_files');
            if (!$oStorageAlbums) {
                bx_log('bx_photos_autoalbum', 'Albums storage not available');
                return;
            }

            $aNewAlbumFileIds = [];

            $oStoragePhotos = BxDolStorage::getObjectInstance('bx_photos_photos');

            foreach ($aQueue as $q) {
                $iQueueId = (int)$q['id'];
                $iPhotoFileId = (int)($aPhotoMap[$iQueueId] ?? 0);
                if (!$iPhotoFileId) {
                    $aFailedQueueIds[] = $iQueueId;
                    continue;
                }

                $iNewId = 0;
                $aPhotoFile = $oStoragePhotos ? $oStoragePhotos->getFile($iPhotoFileId) : null;
                if ($aPhotoFile && !empty($aPhotoFile['path'])) {
                    $sLocalPath = BX_DIRECTORY_PATH_ROOT . 'storage/bx_photos_photos/' . $aPhotoFile['path'];
                    if (file_exists($sLocalPath) && is_readable($sLocalPath)) {
                        $iNewId = (int)$oStorageAlbums->storeFileFromPath($sLocalPath, false, $iProfileId, $iAlbumId, 0, $aPhotoFile['mime_type']);
                    }
                }

                if (!$iNewId) {
                    $iNewId = (int)$oStorageAlbums->storeFileFromStorage([
                        'id' => $iPhotoFileId,
                        'storage' => 'bx_photos_photos',
                        'mime_type' => $aPhotoFile['mime_type'] ?? '',
                    ], false, $iProfileId, $iAlbumId);
                }

                if (!$iNewId) {
                        $sErr = $oStorageAlbums->getErrorString();
                    $iErr = method_exists($oStorageAlbums, 'getErrorCode') ? $oStorageAlbums->getErrorCode() : 0;
                    bx_log('bx_photos_autoalbum', 'Failed to copy photo file into albums storage: src=' . $iPhotoFileId . ' error=' . $sErr . ' code=' . $iErr);
                    $aFailedQueueIds[] = $iQueueId;
                    continue;
                }

             $this->associateFileWithAlbum($iAlbumId, $iNewId, $iProfileId, $aPhotoFile['mime_type']);
                $aNewAlbumFileIds[] = $iNewId;
                $aSuccessQueueIds[] = $iQueueId;
            }

            bx_log('bx_photos_autoalbum', 'Created album #' . $iAlbumId . ' and associated files: ' . json_encode($aNewAlbumFileIds));

            // 7) Delete the original photo entries from the database and storage
            $aPhotoIdsToDelete = array_values($aValidIds);
            if (!empty($aPhotoIdsToDelete)) {
                $oModule = BxDolModule::getInstance('bx_photos');
                if ($oModule) {
                    // Delete the original photo entries that were successfully moved to the album
                    $oModule->_oDb->query("DELETE FROM `bx_photos_photos` WHERE `id` IN (" . $oModule->_oDb->implode_escape($aPhotoIdsToDelete) . ")");
                    
                    // Delete the files from the original storage (bx_photos_photos)
                    $oStoragePhotos->deleteFiles($aPhotoIdsToDelete);
                    
                    bx_log('bx_photos_autoalbum', 'Deleted original photo entries: ' . json_encode($aPhotoIdsToDelete));
                }
            }
            
            // 8) Fire a notification alert (optional)
            if (!empty($aNewAlbumFileIds)) {
                bx_alert('bx_timeline', 'auto_album_created', $iAlbumId, $iProfileId, [
                    'file_ids' => $aNewAlbumFileIds,
                ]);
            }

            // 9) Remove queued items we just processed
            if (!empty($aSuccessQueueIds)) {
                $this->deleteQueueEntries($aSuccessQueueIds);
            }
            if (!empty($aFailedQueueIds)) {
                bx_log('bx_photos_autoalbum', 'Queue entries left for retry: ' . json_encode($aFailedQueueIds));
            }
        } finally {
            $this->releaseLock($iProfileId);
        }
    }

    /**
     * Resolve recent uploads into bx_photos_photos IDs.
     * Uses (profile_id, file_name, size) and recency heuristic.
     * @return int[]
     */
    protected function resolveQueuedPhotoFileIds(int $iProfileId, array $aQueue): array
    {
        $oDb = BxDolDb::getInstance();
        $aMap = [];

        foreach ($aQueue as $q) {
            $iQueueId = (int)$q['id'];
            $sName = (string)$q['name'];
            $iSize = (int)$q['size'];

            $sQuery = $oDb->prepare("SELECT `id` FROM `bx_photos_photos` WHERE `profile_id` = ? AND `file_name` = ? AND `size` = ? ORDER BY `id` DESC LIMIT 1", $iProfileId, $sName, $iSize);
            $iId = (int)$oDb->getOne($sQuery);
            bx_log('bx_photos_autoalbum', "resolveQueuedPhotoFileIds: Query={$sQuery}, Result={$iId}");

            $aMap[$iQueueId] = $iId;
        }

        return $aMap;
    }

    /**
     * Ensure queue table exists.
     */
    protected function ensureQueueTable(): void
    {
        static $done = false;
        if ($done) return;
        $oDb = BxDolDb::getInstance();
        $oDb->query("CREATE TABLE IF NOT EXISTS `bx_photos_autoalbum_queue` (
            `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
            `profile_id` INT UNSIGNED NOT NULL,
            `name` VARCHAR(255) NOT NULL,
            `size` BIGINT UNSIGNED NOT NULL,
            `ts` INT UNSIGNED NOT NULL,
            PRIMARY KEY (`id`),
            KEY `p_ts` (`profile_id`,`ts`)
        ) ENGINE=MyISAM DEFAULT CHARSET=utf8mb4");
        $done = true;
    }

    protected function enqueueUpload(int $iProfileId, string $sName, int $iSize, int $iTs): void
    {
        $oDb = BxDolDb::getInstance();
        $oDb->query($oDb->prepare(
            "INSERT INTO `bx_photos_autoalbum_queue` SET `profile_id`=?, `name`=?, `size`=?, `ts`=?",
            $iProfileId, $sName, $iSize, $iTs
        ));
    }

    protected function fetchRecentQueue(int $iProfileId, int $iSeconds): array
    {
        $oDb = BxDolDb::getInstance();
        $iMinTs = time() - max(1, $iSeconds);
        return (array)$oDb->getAll($oDb->prepare(
            "SELECT `id`,`name`,`size`,`ts` FROM `bx_photos_autoalbum_queue` WHERE `profile_id`=? AND `ts`>=? ORDER BY `id` ASC",
            $iProfileId, $iMinTs
        ));
    }

    protected function deleteQueueEntries(array $aIds): void
    {
        if (empty($aIds)) return;
        $oDb = BxDolDb::getInstance();
        $in = implode(',', array_map('intval', $aIds));
        $oDb->query("DELETE FROM `bx_photos_autoalbum_queue` WHERE `id` IN ($in)");
    }

    protected function determineAlbumCategory(int $iProfileId): string
    {
        $oDb = BxDolDb::getInstance();

        $sLast = (string)$oDb->getOne($oDb->prepare(
            "SELECT `album_category` FROM `bx_albums_albums` WHERE `author`=? AND `album_category`<>'' ORDER BY `id` DESC LIMIT 1",
            $iProfileId
        ));
        if ($sLast !== '') {
            return $sLast;
        }

        $sFirst = (string)$oDb->getOne($oDb->prepare(
            "SELECT `Value` FROM `sys_form_pre_values` WHERE `Key`=? AND `Value`<>'' AND `Value`<>'0' ORDER BY `Order` ASC LIMIT 1",
            'ai-tools'
        ));
        if ($sFirst === '') {
            $sFirst = '1';
        }

        return $sFirst;
    }

    protected function ensureLockTable(): void
    {
        static $done = false;
        if ($done) return;
        $oDb = BxDolDb::getInstance();
        $oDb->query("CREATE TABLE IF NOT EXISTS `bx_photos_autoalbum_lock` (
            `profile_id` INT UNSIGNED NOT NULL,
            `ts` INT UNSIGNED NOT NULL,
            PRIMARY KEY (`profile_id`)
        ) ENGINE=MyISAM DEFAULT CHARSET=utf8mb4");
        $done = true;
    }

    protected function acquireLock(int $iProfileId): bool
    {
        $oDb = BxDolDb::getInstance();
        // INSERT IGNORE style via MyISAM: emulate by checking first, then inserting
        $iExisting = (int)$oDb->getOne($oDb->prepare("SELECT 1 FROM `bx_photos_autoalbum_lock` WHERE `profile_id`=? LIMIT 1", $iProfileId));
        if ($iExisting) return false;
        return (bool)$oDb->query($oDb->prepare("INSERT INTO `bx_photos_autoalbum_lock` SET `profile_id`=?, `ts`=?", $iProfileId, time()));
    }

    protected function releaseLock(int $iProfileId): void
    {
        $oDb = BxDolDb::getInstance();
        $oDb->query($oDb->prepare("DELETE FROM `bx_photos_autoalbum_lock` WHERE `profile_id`=?", $iProfileId));
    }
    /**
     * Check if an auto-album was recently created for this profile.
     */
    protected function recentAutoAlbumExists(int $iProfileId, int $iSeconds): bool
    {
        $oDb = BxDolDb::getInstance();
        $iMinTs = time() - max(1, $iSeconds);
        $sLike = $oDb->escape(_t('_bx_photos_autoalbum_upload_title', '%')); // not ideal for i18n, best-effort
        $iId = (int)$oDb->getOne($oDb->prepare(
            "SELECT `id` FROM `bx_albums_albums` WHERE `author`=? AND `added`>=? ORDER BY `id` DESC LIMIT 1",
            $iProfileId, $iMinTs
        ));
        return $iId > 0;
    }

    /**
     * Associate a file from bx_albums_files with album content.
     */
    protected function associateFileWithAlbum(int $iAlbumId, int $iFileId, int $iProfileId, string $sTitle): void
    {
        $oDb = BxDolDb::getInstance();

        // Determine next order for this album
        $iOrder = (int)$oDb->getOne($oDb->prepare(
            "SELECT IFNULL(MAX(`order`), 0) + 1 FROM `bx_albums_files2albums` WHERE `content_id` = ?",
            $iAlbumId
        ));

        // Insert association
        $oDb->query($oDb->prepare(
            "INSERT INTO `bx_albums_files2albums` SET `content_id` = ?, `file_id` = ?, `author` = ?, `title` = ?, `data` = ?, `exif` = ?, `order` = ? ON DUPLICATE KEY UPDATE `title` = VALUES(`title`), `data` = VALUES(`data`), `exif` = VALUES(`exif`)",
            $iAlbumId, $iFileId, $iProfileId, $sTitle, '', '', $iOrder
        ));
    }
}
